﻿using Santec.Hardware.Devices.Backreflection;
using Santec.Hardware.Devices.Switching;
using Santec.Hardware.Services;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Net;
using System.Runtime.Remoting.Channels;
using System.Text;
using System.Threading.Tasks;

namespace BRMSample
{
    public class Program
    {

        public static async Task Main(string[] args)
        {
            List<IBRM> detectedBRMs;
            IBRM brm;

            IHardwareDetectionService hardwareDetectionService = new HardwareDetectionService();

            double reading;

            // Find the BRM
            Console.WriteLine("Finding USB BRM...");

            // Step 1. Find all the USB BRMs
            detectedBRMs = await hardwareDetectionService.DetectHardwareAsync<IBRM>(EConnectionTypesToDetect.USB);

            if (!detectedBRMs.Any())
            {
                throw new Exception("Could not find BRM");
            }

            // Step 2. Select the first BRM to use for the sample
            brm = detectedBRMs.First();

            Console.WriteLine("BRM Found. Initializing...");

            // Step 3. Initialize the BRM
            await brm.InitializeAsync();

            Console.WriteLine("BRM Initialized.");

            // Step 4. Set the BRM to remote mode
            await brm.SetInteractionModeAsync(Santec.Hardware.Devices.General.EInteractionMode.Remote);

            Console.WriteLine("BRM Interaction Mode set to Remote.");

            // Reference the IL for BRM
            Console.WriteLine("Press any key to reference the IL on Detector 1...");
            _ = Console.ReadKey();
            Console.WriteLine();

            // BRM must have at least one detector to reference IL
            if (brm.NumberOfDetectors < 1)
            {
                Console.WriteLine("BRM has no detectors. Skipping IL.");
            }
            else
            {
                // If the BRM has more than one output, we need to reference the IL at each wavelength for each output channel
                if (brm.NumberOfOutputs > 1)
                {
                    for (int channel = 1; channel <= brm.NumberOfOutputs; channel++)
                    {
                        // Step 1. Switch to the appropriate output channel
                        await brm.ChangeOutputChannelAsync(channel);

                        Console.WriteLine($"Switched BRM to output channel {channel}.");

                        // Step 2. Reference the IL at each wavelength
                        foreach (int wavelength in brm.Wavelengths)
                        {
                            reading = await brm.ReferenceILAsync(1, wavelength);

                            Console.WriteLine($"Referenced IL @ {wavelength}nm: {reading}");
                        }

                        Console.WriteLine();
                    }
                }
                else
                {
                    // If the BRM has a single output, we can reference the IL at each wavelength without changing the BRM output channel
                    // Step 1. Reference the IL at each wavelength
                    foreach (int wavelength in brm.Wavelengths)
                    {
                        reading = await brm.ReferenceILAsync(1, wavelength);

                        Console.WriteLine($"Referenced IL @ {wavelength}nm: {reading}");
                    }

                    Console.WriteLine();
                }
            }

            // Save IL References if needed so that the IL references persist after reboot
            if(brm.RequiresILReferenceSave)
            {
                await brm.SaveILReferencesAsync();
            }

            // Reference BR
            Console.WriteLine("Referencing BR...");

            // If the BRM has more than one output, we need to reference the BR at each wavelength for each output channel
            if (brm.NumberOfOutputs > 1)
            {
                for (int channel = 1; channel <= brm.NumberOfOutputs; channel++)
                {
                    // Step 1. Switch to the appropriate output channel
                    await brm.ChangeOutputChannelAsync(channel);

                    Console.WriteLine($"Switched BRM to output channel {channel}.");

                    Console.WriteLine($"Mandrel wrap the fiber to perform the BR reference.");

                    Console.WriteLine("Press any key to continue...");
                    _ = Console.ReadKey();
                    Console.WriteLine();

                    // Step 2. Reference the BR at each wavelength
                    foreach (int wavelength in brm.Wavelengths)
                    {
                        reading = await brm.ReferenceBRAsync(wavelength);

                        Console.WriteLine($"Referenced BR @ {wavelength}nm: {reading}");
                    }

                    Console.WriteLine();
                }
            }
            else
            {
                // If the BRM has a single output, we can reference the BR at each wavelength without changing the BRM output channel
                // Step 1. Reference the BR at each wavelength
                Console.WriteLine($"Mandrel wrap the fiber to perform the BR reference.");

                Console.WriteLine("Press any key to continue...");
                _ = Console.ReadKey();
                Console.WriteLine();

                foreach (int wavelength in brm.Wavelengths)
                {
                    reading = await brm.ReferenceBRAsync(wavelength);

                    Console.WriteLine($"Referenced BR @ {wavelength}nm: {reading}");
                }

                Console.WriteLine();
            }

            // Measure IL and BR
            if (brm.NumberOfDetectors < 1)
            {
                Console.WriteLine("Done referencing BR.");

                Console.WriteLine("Press any key to measure BR...");
            }
            else
            {
                Console.WriteLine("Done referencing IL on Detector 1 and BR.");

                Console.WriteLine("Press any key to measure IL and BR...");
            }

            _ = Console.ReadKey();
            Console.WriteLine();

            // If the BRM has more than one output, we need to measure the IL and BR at each wavelength for each output channel
            if(brm.NumberOfOutputs > 1)
            {
                for (int channel = 1; channel <= brm.NumberOfOutputs; channel++)
                {
                    // Step 1. Switch to the BRM to the appropriate output channel
                    await brm.ChangeOutputChannelAsync(channel);

                    Console.WriteLine($"Switched BRM to output channel {channel}.");

                    foreach (int wavelength in brm.Wavelengths)
                    {
                        // Step 2. Measure IL if there is a detector
                        if (brm.NumberOfDetectors > 0)
                        {
                            reading = await brm.ReadILAsync(1, wavelength);

                            Console.WriteLine($"Measured IL @ {wavelength}nm: {reading}");
                        }

                        // Step 3. Measure BR
                        reading = (await brm.ReadBRAsync(wavelength)).Value;

                        Console.WriteLine($"Measured BR @ {wavelength}nm: {reading}");
                    }

                    Console.WriteLine();
                }
            }
            else
            {
                // If the BRM has a single output, we can measure the IL and BR at each wavelength without changing the output channel
                foreach (int wavelength in brm.Wavelengths)
                {
                    // Step 1. Measure IL if there is a detector
                    if (brm.NumberOfDetectors > 0)
                    {
                        reading = await brm.ReadILAsync(1, wavelength);

                        Console.WriteLine($"Measured IL @ {wavelength}nm: {reading}");
                    }

                    // Step 2. Measure BR
                    reading = (await brm.ReadBRAsync(wavelength)).Value;

                    Console.WriteLine($"Measured BR @ {wavelength}nm: {reading}");
                }

                Console.WriteLine();
            }

            if (brm.NumberOfDetectors < 1)
            {
                Console.WriteLine("Done measuring BR.");
            }
            else
            {
                Console.WriteLine("Done measuring IL and BR.");
            }

            // Cleanup
            // Step 1. Uninitialize the BRM
            brm.Uninitialize();

            Console.WriteLine("Press any key to exit...");
            _ = Console.ReadKey();
        }
    }
}